<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;

/**
 * Command ValidateProductionConfig
 * 
 * Validasi konfigurasi keamanan untuk production deployment
 * Memastikan semua security settings sudah benar
 */
class ValidateProductionConfig extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'config:validate-production 
                            {--fix : Attempt to fix issues automatically}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Validasi konfigurasi keamanan untuk production deployment';

    /**
     * Validation results
     */
    protected $passed = [];
    protected $warnings = [];
    protected $failed = [];

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('===========================================');
        $this->info('  RDKK Security Configuration Validator');
        $this->info('===========================================');
        $this->newLine();

        // Run all checks
        $this->checkEnvironment();
        $this->checkAppKey();
        $this->checkDebugMode();
        $this->checkHttps();
        $this->checkSession();
        $this->checkDatabase();
        $this->checkLogging();
        $this->checkFilePermissions();

        // Display results
        $this->displayResults();

        // Return exit code
        return count($this->failed) > 0 ? 1 : 0;
    }

    /**
     * Check environment configuration
     */
    protected function checkEnvironment()
    {
        $env = config('app.env');
        
        if ($env === 'production') {
            $this->passed[] = 'APP_ENV set to production';
        } else {
            $this->warnings[] = "APP_ENV is '{$env}', should be 'production' for production deployment";
        }
    }

    /**
     * Check APP_KEY is set
     */
    protected function checkAppKey()
    {
        $key = config('app.key');
        
        if (empty($key)) {
            $this->failed[] = 'APP_KEY is not set! Run: php artisan key:generate';
        } else {
            $this->passed[] = 'APP_KEY is configured';
        }
    }

    /**
     * Check debug mode
     */
    protected function checkDebugMode()
    {
        $debug = config('app.debug');
        
        if ($debug === true) {
            $this->failed[] = 'APP_DEBUG is TRUE! Must be FALSE in production (security risk!)';
        } else {
            $this->passed[] = 'APP_DEBUG is disabled';
        }
    }

    /**
     * Check HTTPS configuration
     */
    protected function checkHttps()
    {
        $url = config('app.url');
        
        if (strpos($url, 'https://') === 0) {
            $this->passed[] = 'APP_URL uses HTTPS';
        } else {
            $this->warnings[] = 'APP_URL should use HTTPS in production';
        }
    }

    /**
     * Check session configuration
     */
    protected function checkSession()
    {
        // Session encryption
        $encrypt = config('session.encrypt');
        if ($encrypt === true) {
            $this->passed[] = 'Session encryption is enabled';
        } else {
            $this->failed[] = 'Session encryption is DISABLED! Must be enabled for sensitive data';
        }

        // Session driver
        $driver = config('session.driver');
        if ($driver === 'database') {
            $this->passed[] = 'Session driver is database (good for audit trail)';
        } else {
            $this->warnings[] = "Session driver is '{$driver}', consider 'database' for better audit trail";
        }

        // Session lifetime
        $lifetime = config('session.lifetime');
        if ($lifetime <= 30) {
            $this->passed[] = "Session lifetime is {$lifetime} minutes (secure)";
        } else {
            $this->warnings[] = "Session lifetime is {$lifetime} minutes, recommended ≤ 30 minutes for sensitive data";
        }

        // Secure cookie
        $secure = config('session.secure');
        if ($secure === true) {
            $this->passed[] = 'Session cookies set to secure (HTTPS only)';
        } else {
            $this->warnings[] = 'SESSION_SECURE_COOKIE should be true in production';
        }

        // HTTP only
        $httpOnly = config('session.http_only');
        if ($httpOnly === true) {
            $this->passed[] = 'Session cookies are HTTP only';
        } else {
            $this->failed[] = 'Session cookies must be HTTP only (security risk!)';
        }
    }

    /**
     * Check database configuration
     */
    protected function checkDatabase()
    {
        $connection = config('database.default');
        $config = config("database.connections.{$connection}");

        // Strict mode
        if (isset($config['strict']) && $config['strict'] === true) {
            $this->passed[] = 'Database strict mode is enabled';
        } else {
            $this->warnings[] = 'Database strict mode should be enabled for data integrity';
        }

        // SSL configuration
        if (isset($config['options']) && is_array($config['options'])) {
            if (isset($config['options'][\PDO::MYSQL_ATTR_SSL_CA])) {
                $this->passed[] = 'Database SSL is configured';
            } else {
                $this->warnings[] = 'Consider enabling SSL for database connections in production';
            }
        }
    }

    /**
     * Check logging configuration
     */
    protected function checkLogging()
    {
        $level = config('logging.channels.daily.level');
        
        if (in_array($level, ['warning', 'error', 'critical'])) {
            $this->passed[] = "Log level is '{$level}' (appropriate for production)";
        } else {
            $this->warnings[] = "Log level is '{$level}', consider 'warning' or 'error' for production";
        }

        // Log retention
        $days = config('logging.channels.daily.days');
        if ($days >= 30) {
            $this->passed[] = "Log retention is {$days} days (good for audit)";
        } else {
            $this->warnings[] = "Log retention is {$days} days, recommended ≥ 30 days for audit trail";
        }
    }

    /**
     * Check critical file permissions
     */
    protected function checkFilePermissions()
    {
        $files = [
            '.env' => 0600,
            'storage' => 0755,
            'bootstrap/cache' => 0755,
        ];

        foreach ($files as $path => $expectedPerm) {
            $fullPath = base_path($path);
            
            if (!file_exists($fullPath)) {
                $this->warnings[] = "File/directory not found: {$path}";
                continue;
            }

            $perms = fileperms($fullPath) & 0777;
            
            if ($path === '.env') {
                // .env should be restrictive (600 or 400)
                if ($perms === 0600 || $perms === 0400) {
                    $this->passed[] = ".env permissions are secure (" . decoct($perms) . ")";
                } else {
                    $this->warnings[] = ".env permissions are " . decoct($perms) . ", should be 600 or 400";
                }
            } else {
                if ($perms === $expectedPerm) {
                    $this->passed[] = "{$path} permissions are correct (" . decoct($perms) . ")";
                } else {
                    $this->warnings[] = "{$path} permissions are " . decoct($perms) . ", expected " . decoct($expectedPerm);
                }
            }
        }
    }

    /**
     * Display validation results
     */
    protected function displayResults()
    {
        $this->newLine();
        $this->info('===========================================');
        $this->info('  Validation Results');
        $this->info('===========================================');
        $this->newLine();

        // Passed checks
        if (count($this->passed) > 0) {
            $this->info('✓ PASSED CHECKS (' . count($this->passed) . '):');
            foreach ($this->passed as $check) {
                $this->line('  ✓ ' . $check);
            }
            $this->newLine();
        }

        // Warnings
        if (count($this->warnings) > 0) {
            $this->warn('⚠ WARNINGS (' . count($this->warnings) . '):');
            foreach ($this->warnings as $warning) {
                $this->line('  ⚠ ' . $warning);
            }
            $this->newLine();
        }

        // Failed checks
        if (count($this->failed) > 0) {
            $this->error('✗ FAILED CHECKS (' . count($this->failed) . '):');
            foreach ($this->failed as $fail) {
                $this->line('  ✗ ' . $fail);
            }
            $this->newLine();
        }

        // Summary
        $total = count($this->passed) + count($this->warnings) + count($this->failed);
        $this->info('===========================================');
        $this->info("Total Checks: {$total}");
        $this->info("Passed: " . count($this->passed));
        $this->warn("Warnings: " . count($this->warnings));
        if (count($this->failed) > 0) {
            $this->error("Failed: " . count($this->failed));
            $this->newLine();
            $this->error('❌ PRODUCTION DEPLOYMENT NOT RECOMMENDED!');
            $this->error('Please fix failed checks before deploying to production.');
        } else {
            $this->newLine();
            $this->info('✅ Configuration is ready for production!');
            if (count($this->warnings) > 0) {
                $this->warn('Consider addressing warnings for optimal security.');
            }
        }
        $this->info('===========================================');
    }
}

