<?php

namespace App\Exports;

use App\Models\Poktan;
use App\Models\Desa;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\WithStyles;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use Maatwebsite\Excel\Concerns\WithEvents;
use Maatwebsite\Excel\Events\AfterSheet;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;
use PhpOffice\PhpSpreadsheet\Style\Fill;
use PhpOffice\PhpSpreadsheet\Style\Color;
use PhpOffice\PhpSpreadsheet\Style\Alignment;

class PoktanExport implements FromCollection, WithHeadings, WithMapping, WithStyles, ShouldAutoSize, WithEvents
{
    protected $isTemplate;

    public function __construct($isTemplate = false)
    {
        $this->isTemplate = $isTemplate;
    }

    /**
    * @return \Illuminate\Support\Collection
    */
    public function collection()
    {
        if ($this->isTemplate) {
            // Return collection with example data for template
            $exampleDesa = Desa::first();
            
            return collect([
                (object)[
                    'id_poktan' => 'PKT001',
                    'nama_poktan' => 'Contoh Kelompok Tani Makmur',
                    'kode_desa' => $exampleDesa ? $exampleDesa->kode_desa : '0000000000',
                ]
            ]);
        }
        
        return Poktan::with('desa')->get();
    }

    /**
     * @return array
     */
    public function headings(): array
    {
        if ($this->isTemplate) {
            return [            
                'ID Poktan',
                'Nama Poktan',
                'Kode Desa',
            ];
        }
        
        return [            
            'ID Poktan',
            'Nama Poktan',
            'Kode Desa',
            'Nama Desa',
            'Tanggal Dibuat',
            'Tanggal Diperbarui'
        ];
    }

    /**
     * @param mixed $row
     * @return array
     */
    public function map($row): array
    {
        if ($this->isTemplate) {
            return [
                $row->id_poktan,
                $row->nama_poktan,
                $row->kode_desa,
            ];
        }
        
        return [
            $row->id_poktan,
            $row->nama_poktan,
            $row->kode_desa,
            $row->desa ? $row->desa->nama_desa : '',
            $row->created_at ? $row->created_at->format('Y-m-d H:i:s') : '',
            $row->updated_at ? $row->updated_at->format('Y-m-d H:i:s') : ''
        ];
    }

    /**
     * @param Worksheet $sheet
     * @return array
     */
    public function styles(Worksheet $sheet)
    {
        $styles = [
            // Style the first row as bold text with background color
            1 => [
                'font' => [
                    'bold' => true,
                    'color' => ['rgb' => 'FFFFFF']
                ],
                'fill' => [
                    'fillType' => Fill::FILL_SOLID,
                    'startColor' => ['rgb' => '4F46E5']
                ],
                'alignment' => [
                    'horizontal' => Alignment::HORIZONTAL_CENTER,
                    'vertical' => Alignment::VERTICAL_CENTER,
                ]
            ],
        ];

        if ($this->isTemplate) {
            // Style example row with different color
            $styles[2] = [
                'fill' => [
                    'fillType' => Fill::FILL_SOLID,
                    'startColor' => ['rgb' => 'E0E7FF']
                ],
                'font' => [
                    'italic' => true,
                    'color' => ['rgb' => '6B7280']
                ]
            ];
        }

        return $styles;
    }

    /**
     * @return array
     */
    public function registerEvents(): array
    {
        return [
            AfterSheet::class => function(AfterSheet $event) {
                if ($this->isTemplate) {
                    $sheet = $event->sheet->getDelegate();
                    
                    // Add instruction comment to first cell
                    $sheet->getComment('A1')->getText()->createTextRun(
                        "TEMPLATE IMPORT KELOMPOK TANI\n\n" .
                        "KOLOM YANG HARUS DIISI:\n" .
                        "1. ID Poktan: Kode unik kelompok tani (maks 20 karakter)\n" .
                        "2. Nama Poktan: Nama kelompok tani (maks 100 karakter)\n" .
                        "3. Kode Desa: Kode desa yang sudah terdaftar di sistem\n\n" .
                        "CATATAN:\n" .
                        "- Baris ke-2 adalah contoh data, hapus sebelum import\n" .
                        "- Pastikan kode desa sudah terdaftar di sistem\n" .
                        "- Jangan mengubah nama kolom header (baris 1)\n" .
                        "- Simpan dalam format .xlsx atau .xls"
                    );
                    
                    // Set comment size
                    $sheet->getComment('A1')->setWidth('300pt');
                    $sheet->getComment('A1')->setHeight('200pt');
                }
            },
        ];
    }
}

