<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;

/**
 * Middleware EnsureUserIsAdmin
 * 
 * Memastikan hanya user dengan role admin yang dapat mengakses route tertentu
 * Untuk proteksi operasi sensitif seperti user management
 */
class EnsureUserIsAdmin
{
    /**
     * Handle an incoming request.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     */
    public function handle(Request $request, Closure $next): Response
    {
        // Cek apakah user sudah terautentikasi
        if (!auth()->check()) {
            abort(401, 'Unauthorized');
        }

        // Cek apakah user memiliki role admin
        if (auth()->user()->role !== 'admin') {
            // Log unauthorized access attempt untuk audit
            \Log::warning('Unauthorized admin access attempt', [
                'user_id' => auth()->id(),
                'user_email' => auth()->user()->email,
                'user_role' => auth()->user()->role,
                'ip_address' => $request->ip(),
                'url' => $request->fullUrl(),
                'timestamp' => now()->format('Y-m-d H:i:s')
            ]);

            abort(403, 'Akses ditolak. Anda tidak memiliki izin untuk mengakses halaman ini.');
        }

        return $next($request);
    }
}

