<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Str;
use App\Traits\EncryptsAttributes;
use Spatie\Activitylog\Traits\LogsActivity;
use Spatie\Activitylog\LogOptions;

/**
 * Class Petani
 * 
 * Model untuk entitas Petani yang menyimpan informasi lengkap petani
 * 
 * @property string $id UUID sebagai primary key
 * @property string $nama_penyuluh Nama penyuluh petani
 * @property string $no_ktp Nomor KTP petani (16 digit)
 * @property string $nama_ibu_kandung Nama ibu kandung petani
 * @property string $kode_desa Kode desa tempat petani berada
 * @property string $id_poktan ID kelompok tani
 * @property string $kode_kios Kode kios yang terkait dengan petani
 * @property string $subsektor Subsektor pertanian (tanaman_pangan, hortikultura, perkebunan, peternakan)
 * @property string $komoditas Jenis komoditas yang ditanam
 * @property float $luas_lahan Luas lahan dalam hektar
 * @property string $status Status petani dalam sistem (belum_terinput, sudah_terinput)
 * @property string $alamat Alamat lengkap petani
 * @property string $no_hp Nomor handphone petani
 * @property string $email Email petani (opsional)
 * @property string $jenis_kelamin Jenis kelamin petani (L/P)
 * @property date $tanggal_lahir Tanggal lahir petani
 * @property string $created_by ID user yang membuat data
 * @property string $updated_by ID user yang mengupdate data
 * @property \Carbon\Carbon $created_at Timestamp pembuatan data
 * @property \Carbon\Carbon $updated_at Timestamp update data
 * @property \Carbon\Carbon $deleted_at Timestamp soft delete
 */
class Petani extends Model
{
    use HasFactory, SoftDeletes, EncryptsAttributes, LogsActivity;
    
    /**
     * Konfigurasi primary key
     */
    protected $primaryKey = 'id';
    public $incrementing = false;
    protected $keyType = 'string';
    
    /**
     * Daftar field yang harus dienkripsi di database
     * Untuk keamanan data sensitif sesuai compliance
     */
    protected $encryptable = [
        // Encryption disabled for these columns to keep within column lengths
        // and preserve deterministic uniqueness constraints at the DB level.
        // Consider using an encrypted storage + deterministic hash column instead.
        // 'no_ktp',
        // 'nama_ibu_kandung',
    ];
    
    /**
     * Daftar field yang dapat diisi (mass assignable)
     */
    protected $fillable = [
        'nama_penyuluh',
        'no_ktp',
        'nama_ibu_kandung',
        'kode_desa',
        'id_poktan',
        'kode_kios',
        'subsektor',
        'komoditas',
        'luas_lahan',
        'status',
        'alamat',
        'no_hp',
        'email',
        'jenis_kelamin',
        'tanggal_lahir',
    ];
    
    /**
     * Daftar field dengan tipe data khusus
     */
    protected $casts = [
        'luas_lahan' => 'decimal:2',
        'tanggal_lahir' => 'date',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];
    
    /**
     * Daftar field yang disembunyikan saat serialisasi
     */
    protected $hidden = [
        'created_by',
        'updated_by',
        'deleted_at',
    ];
    
    /**
     * Boot method untuk mengatur event model
     */
    protected static function boot()
    {
        parent::boot();
        
        // Generate UUID saat pembuatan record baru
        static::creating(function ($model) {
            $model->id = $model->id ?? Str::uuid();
            $model->created_by = auth()->id() ?? null;
        });
        
        // Set user yang melakukan update
        static::updating(function ($model) {
            $model->updated_by = auth()->id() ?? null;
        });
    }
    
    /**
     * Relasi dengan model Desa
     */
    public function desa()
    {
        return $this->belongsTo(Desa::class, 'kode_desa', 'kode_desa');
    }
    
    /**
     * Relasi dengan model Poktan (Kelompok Tani)
     */
    public function poktan()
    {
        return $this->belongsTo(Poktan::class, 'id_poktan', 'id_poktan');
    }
    
    /**
     * Relasi dengan model Kios
     */
    public function kios()
    {
        return $this->belongsTo(Kios::class, 'kode_kios', 'kode_kios');
    }
    
    /**
     * Relasi dengan model User (pembuat data)
     */
    public function createdBy()
    {
        return $this->belongsTo(User::class, 'created_by');
    }
    
    /**
     * Relasi dengan model User (pengupdate data)
     */
    public function updatedBy()
    {
        return $this->belongsTo(User::class, 'updated_by');
    }
    
    /**
     * Scope untuk filter berdasarkan status
     */
    public function scopeStatus($query, $status)
    {
        return $query->where('status', $status);
    }
    
    /**
     * Scope untuk filter berdasarkan subsektor
     */
    public function scopeSubsektor($query, $subsektor)
    {
        return $query->where('subsektor', $subsektor);
    }
    
    /**
     * Configure activity logging
     * SECURITY: Log changes tapi exclude sensitive data dari log
     */
    public function getActivitylogOptions(): LogOptions
    {
        return LogOptions::defaults()
            ->logOnly([
                'nama_penyuluh',
                'kode_desa',
                'id_poktan',
                'kode_kios',
                'subsektor',
                'komoditas',
                'luas_lahan',
                'status',
            ])
            ->logOnlyDirty()
            ->dontSubmitEmptyLogs();
        // Note: no_ktp dan nama_ibu_kandung TIDAK di-log untuk security
    }
}