<?php

namespace App\Services;

use Illuminate\Support\Facades\Log;
use Spatie\Activitylog\Models\Activity;

/**
 * Service ActivityLogger
 * 
 * Centralized service untuk activity logging dan security event monitoring
 * Menggunakan Spatie Activity Log untuk audit trail
 */
class ActivityLogger
{
    /**
     * Log aktivitas CRUD pada data sensitif
     */
    public static function logDataAccess(string $action, string $model, $modelId, array $properties = [])
    {
        activity()
            ->performedOn($model::find($modelId))
            ->withProperties($properties)
            ->log($action);
    }

    /**
     * Log operasi data export
     */
    public static function logExport(string $exportType, int $recordCount = null)
    {
        $properties = [
            'export_type' => $exportType,
            'record_count' => $recordCount,
            'ip_address' => request()->ip(),
            'user_agent' => request()->userAgent(),
        ];

        activity()
            ->withProperties($properties)
            ->log("exported_{$exportType}_data");

        // Log ke security log juga
        Log::channel('security')->info('Data Export', [
            'user_id' => auth()->id(),
            'user_email' => auth()->user()->email ?? 'unknown',
            'export_type' => $exportType,
            'record_count' => $recordCount,
            'ip_address' => request()->ip(),
            'timestamp' => now()->format('Y-m-d H:i:s'),
        ]);
    }

    /**
     * Log operasi data import
     */
    public static function logImport(string $importType, int $recordCount = null, bool $success = true)
    {
        $properties = [
            'import_type' => $importType,
            'record_count' => $recordCount,
            'success' => $success,
            'ip_address' => request()->ip(),
            'user_agent' => request()->userAgent(),
        ];

        activity()
            ->withProperties($properties)
            ->log("imported_{$importType}_data");

        // Log ke security log
        Log::channel('security')->info('Data Import', [
            'user_id' => auth()->id(),
            'user_email' => auth()->user()->email ?? 'unknown',
            'import_type' => $importType,
            'record_count' => $recordCount,
            'success' => $success,
            'ip_address' => request()->ip(),
            'timestamp' => now()->format('Y-m-d H:i:s'),
        ]);
    }

    /**
     * Log failed login attempts
     */
    public static function logFailedLogin(string $email)
    {
        $properties = [
            'email' => $email,
            'ip_address' => request()->ip(),
            'user_agent' => request()->userAgent(),
        ];

        activity()
            ->withProperties($properties)
            ->log('failed_login_attempt');

        // Log ke security log dengan level warning
        Log::channel('security')->warning('Failed Login Attempt', [
            'email' => $email,
            'ip_address' => request()->ip(),
            'user_agent' => request()->userAgent(),
            'timestamp' => now()->format('Y-m-d H:i:s'),
        ]);
    }

    /**
     * Log successful login
     */
    public static function logSuccessfulLogin()
    {
        $properties = [
            'ip_address' => request()->ip(),
            'user_agent' => request()->userAgent(),
        ];

        activity()
            ->withProperties($properties)
            ->log('successful_login');

        Log::channel('security')->info('Successful Login', [
            'user_id' => auth()->id(),
            'user_email' => auth()->user()->email ?? 'unknown',
            'ip_address' => request()->ip(),
            'timestamp' => now()->format('Y-m-d H:i:s'),
        ]);
    }

    /**
     * Log password change
     */
    public static function logPasswordChange($userId = null)
    {
        $userId = $userId ?? auth()->id();

        activity()
            ->withProperties([
                'user_id' => $userId,
                'ip_address' => request()->ip(),
            ])
            ->log('password_changed');

        Log::channel('security')->info('Password Changed', [
            'user_id' => $userId,
            'ip_address' => request()->ip(),
            'timestamp' => now()->format('Y-m-d H:i:s'),
        ]);
    }

    /**
     * Log role changes (critical security event)
     */
    public static function logRoleChange($userId, string $oldRole, string $newRole)
    {
        $properties = [
            'user_id' => $userId,
            'old_role' => $oldRole,
            'new_role' => $newRole,
            'changed_by' => auth()->id(),
            'ip_address' => request()->ip(),
        ];

        activity()
            ->withProperties($properties)
            ->log('user_role_changed');

        // Critical security event
        Log::channel('security')->warning('User Role Changed', [
            'target_user_id' => $userId,
            'old_role' => $oldRole,
            'new_role' => $newRole,
            'changed_by_user_id' => auth()->id(),
            'changed_by_email' => auth()->user()->email ?? 'unknown',
            'ip_address' => request()->ip(),
            'timestamp' => now()->format('Y-m-d H:i:s'),
        ]);
    }

    /**
     * Log user deletion (critical security event)
     */
    public static function logUserDeletion($userId, string $userEmail)
    {
        $properties = [
            'deleted_user_id' => $userId,
            'deleted_user_email' => $userEmail,
            'deleted_by' => auth()->id(),
            'ip_address' => request()->ip(),
        ];

        activity()
            ->withProperties($properties)
            ->log('user_deleted');

        Log::channel('security')->warning('User Deleted', [
            'deleted_user_id' => $userId,
            'deleted_user_email' => $userEmail,
            'deleted_by_user_id' => auth()->id(),
            'deleted_by_email' => auth()->user()->email ?? 'unknown',
            'ip_address' => request()->ip(),
            'timestamp' => now()->format('Y-m-d H:i:s'),
        ]);
    }

    /**
     * Log sensitive data access
     */
    public static function logSensitiveDataAccess(string $dataType, $recordId)
    {
        $properties = [
            'data_type' => $dataType,
            'record_id' => $recordId,
            'ip_address' => request()->ip(),
            'user_agent' => request()->userAgent(),
        ];

        activity()
            ->withProperties($properties)
            ->log('sensitive_data_accessed');

        Log::channel('security')->info('Sensitive Data Accessed', [
            'user_id' => auth()->id(),
            'user_email' => auth()->user()->email ?? 'unknown',
            'data_type' => $dataType,
            'record_id' => $recordId,
            'ip_address' => request()->ip(),
            'timestamp' => now()->format('Y-m-d H:i:s'),
        ]);
    }

    /**
     * Get recent activities untuk dashboard admin
     */
    public static function getRecentActivities(int $limit = 50)
    {
        return Activity::with('causer')
            ->latest()
            ->limit($limit)
            ->get();
    }

    /**
     * Get security events (failed logins, role changes, etc)
     */
    public static function getSecurityEvents(int $limit = 50)
    {
        return Activity::whereIn('description', [
                'failed_login_attempt',
                'user_role_changed',
                'user_deleted',
                'sensitive_data_accessed',
            ])
            ->latest()
            ->limit($limit)
            ->get();
    }
}

