# Security Implementation Guide - Aplikasi RDKK

## 📋 Overview

Dokumen ini merangkum semua implementasi keamanan yang telah diterapkan pada aplikasi RDKK. Implementasi ini mencakup enkripsi data sensitif, access control, audit logging, security headers, dan berbagai proteksi lainnya sesuai standar keamanan tinggi.

---

## ✅ Checklist Implementasi Keamanan

### 1. Enkripsi Data Sensitif

- [x] **Database Encryption**
  - File: `app/Traits/EncryptsAttributes.php`
  - Implemented: AES-256 encryption menggunakan Laravel Crypt
  - Data encrypted: `no_ktp`, `nama_ibu_kandung`
  - Model: `app/Models/Petani.php`

- [x] **Session Encryption**
  - Config: `config/session.php`
  - Session encryption: Enabled
  - Session driver: Database (untuk audit trail)
  - Session lifetime: 30 menit

### 2. Authentication & Authorization

- [x] **Admin Middleware**
  - File: `app/Http/Middleware/EnsureUserIsAdmin.php`
  - Registered: `app/Http/Kernel.php`
  - Applied: `routes/web.php` (user management routes)

- [x] **Enhanced Authentication**
  - Session timeout: 30 menit
  - Password confirmation timeout: 30 menit (1800 detik)
  - Config: `config/auth.php`, `config/session.php`

- [x] **Strong Password Policy**
  - File: `app/Rules/StrongPassword.php`
  - Requirements:
    - Minimal 12 karakter
    - 1 uppercase, 1 lowercase, 1 number, 1 symbol
    - Check common passwords
  - Applied: `app/Http/Controllers/UserController.php`

- [x] **Secure Admin Seeder**
  - File: `database/seeders/AdminSeeder.php`
  - Default password: `Rdkk@Admin2024!Secure`
  - ⚠️ **MUST CHANGE** after first login

### 3. Security Headers & HTTPS

- [x] **Security Headers Middleware**
  - File: `app/Http/Middleware/SecurityHeaders.php`
  - Headers implemented:
    - Content-Security-Policy (CSP)
    - Strict-Transport-Security (HSTS)
    - X-Frame-Options: DENY
    - X-Content-Type-Options: nosniff
    - X-XSS-Protection
    - Referrer-Policy
    - Permissions-Policy

- [x] **HTTPS Enforcement**
  - File: `app/Http/Middleware/ForceHttps.php`
  - Auto-redirect ke HTTPS di production
  - Secure cookies enabled

- [x] **CORS Configuration**
  - File: `config/cors.php`
  - Allowed origins: restricted to APP_URL
  - Credentials support: enabled
  - Methods: specified (not wildcard)

### 4. Input Validation & Sanitization

- [x] **Enhanced File Upload**
  - Max size: 2MB untuk Excel import
  - MIME validation: xlsx, xls only
  - Location: `app/Http/Controllers/PetaniController.php`

### 5. Audit Logging & Monitoring

- [x] **Activity Logger Service**
  - File: `app/Services/ActivityLogger.php`
  - Spatie Activity Log integration
  - Models configured: `User`, `Petani`

- [x] **Security Event Logging**
  - Failed login attempts
  - Successful logins
  - Password changes
  - Role changes
  - User deletions
  - Data export/import
  - Sensitive data access

- [x] **Log Configuration**
  - File: `config/logging.php`
  - Application log: 90 days retention
  - Security log: 365 days retention
  - Production level: warning
  - Separate security channel created

- [x] **Sensitive Data Masking**
  - `no_ktp` removed from logs
  - Password never logged
  - Files: `app/Services/PetaniService.php`

### 6. Database Security

- [x] **Database Hardening**
  - File: `config/database.php`
  - Strict mode: enabled
  - SSL support: configured
  - Prepared statements: enabled via PDO settings

### 7. Backup & Recovery

- [x] **Backup Command**
  - File: `app/Console/Commands/BackupDatabase.php`
  - Features:
    - Database dump
    - Compression (--compress)
    - Encryption (--encrypt)
    - Auto cleanup (30 days)
  - Usage: `php artisan backup:database --encrypt`

### 8. Error Handling

- [x] **Custom Error Pages**
  - Files:
    - `resources/views/errors/403.blade.php`
    - `resources/views/errors/404.blade.php`
    - `resources/views/errors/419.blade.php`
    - `resources/views/errors/500.blade.php`
    - `resources/views/errors/503.blade.php`
  - No sensitive information disclosure

### 9. Rate Limiting

- [x] **Enhanced Rate Limiting**
  - Export: 5 requests/minute
  - Import: 3 requests/minute
  - Login: 5 attempts (built-in Laravel)
  - Location: `routes/web.php`

### 10. Configuration & Deployment

- [x] **Production Config Validator**
  - File: `app/Console/Commands/ValidateProductionConfig.php`
  - Checks:
    - APP_ENV
    - APP_DEBUG
    - APP_KEY
    - HTTPS
    - Session config
    - Database config
    - Logging config
    - File permissions
  - Usage: `php artisan config:validate-production`

### 11. Documentation

- [x] **Security Policy**
  - File: `SECURITY.md`
  - Comprehensive security documentation
  - Incident response plan
  - Compliance guidelines

- [x] **Implementation Guide**
  - File: `docs/SECURITY_IMPLEMENTATION_GUIDE.md` (this file)

---

## 🚀 Deployment Checklist

### Pre-Deployment

1. **Environment Configuration**
   ```bash
   # Copy .env.example ke .env
   cp .env.example .env
   
   # Generate application key
   php artisan key:generate
   
   # Set environment variables
   APP_ENV=production
   APP_DEBUG=false
   APP_URL=https://your-domain.com
   SESSION_SECURE_COOKIE=true
   SESSION_ENCRYPT=true
   LOG_LEVEL=warning
   ```

2. **Database Setup**
   ```bash
   # Run migrations
   php artisan migrate
   
   # Seed admin user
   php artisan db:seed --class=AdminSeeder
   
   # Publish activity log migration
   php artisan vendor:publish --provider="Spatie\Activitylog\ActivitylogServiceProvider" --tag="activitylog-migrations"
   php artisan migrate
   ```

3. **Validate Configuration**
   ```bash
   php artisan config:validate-production
   ```

4. **Change Default Password**
   - Login dengan admin@rdkk.com / Rdkk@Admin2024!Secure
   - Segera ganti password!

### Post-Deployment

1. **Verify Security Headers**
   ```bash
   curl -I https://your-domain.com
   ```
   Check for security headers

2. **Test Authentication**
   - Test login/logout
   - Test failed login (check logs)
   - Test session timeout

3. **Test Encryption**
   - Add petani data
   - Check database (data harus encrypted)
   - View di aplikasi (data harus decrypted)

4. **Configure Backup**
   ```bash
   # Test manual backup
   php artisan backup:database --encrypt
   
   # Add ke cron untuk automated backup
   # Add to crontab:
   0 2 * * * cd /path/to/app && php artisan backup:database --encrypt
   ```

5. **Monitor Logs**
   ```bash
   # Application logs
   tail -f storage/logs/laravel.log
   
   # Security logs
   tail -f storage/logs/security.log
   ```

---

## 🔍 Testing Security Implementation

### 1. Test Enkripsi

```bash
# Add data via aplikasi
# Check database
mysql> SELECT no_ktp FROM petanis LIMIT 1;
# Seharusnya muncul encrypted string

# View via aplikasi
# Seharusnya muncul decrypted data
```

### 2. Test Access Control

```bash
# Login sebagai regular user
# Try to access /user (user management)
# Should get 403 Forbidden

# Check security.log untuk unauthorized access attempt
```

### 3. Test Rate Limiting

```bash
# Test export
for i in {1..10}; do
  curl -H "Cookie: your_session_cookie" https://your-domain.com/petani-export
done
# Should get rate limited after 5 requests

# Test failed login
for i in {1..10}; do
  curl -X POST https://your-domain.com/login \
    -d "email=test@test.com&password=wrong"
done
# Should get locked after 5 attempts
```

### 4. Test Security Headers

```bash
curl -I https://your-domain.com | grep -E "(X-Frame|X-Content|Strict-Transport|Content-Security)"
```

### 5. Test Audit Logging

```bash
# Perform various actions
# Check activity log
php artisan tinker
>>> \Spatie\Activitylog\Models\Activity::latest()->limit(10)->get()

# Check security log
tail -f storage/logs/security.log
```

---

## 🔧 Maintenance

### Regular Tasks

**Daily:**
- Monitor security logs untuk suspicious activities
- Check failed login attempts

**Weekly:**
- Review activity logs
- Check disk space untuk logs dan backups
- Verify automated backups berhasil

**Monthly:**
- Update dependencies
  ```bash
  composer audit
  composer update
  ```
- Review user access list
- Test backup restore procedure

**Quarterly:**
- Security audit
- Review dan update security policies
- Penetration testing (if required)
- Password rotation (admin accounts)

### Troubleshooting

**Issue: Dekripsi gagal**
```bash
# Check APP_KEY
php artisan key:generate --show

# Clear config cache
php artisan config:clear
php artisan cache:clear
```

**Issue: Session expired terlalu cepat**
```env
# Adjust di .env
SESSION_LIFETIME=30  # minutes
```

**Issue: Too many logs**
```bash
# Clean old logs manually
find storage/logs -name "*.log" -mtime +90 -delete

# Atau gunakan Laravel log rotation (sudah configured)
```

**Issue: Backup gagal**
```bash
# Check permissions
ls -la storage/app/backups/

# Check mysqldump available
which mysqldump

# Test manually
mysqldump -u user -p database > test_backup.sql
```

---

## 📞 Support & Contact

**Security Issues:**
- Email: security@rdkk.com
- Lihat `SECURITY.md` untuk responsible disclosure

**Technical Support:**
- [Your support contact]

**Documentation:**
- Security Policy: `SECURITY.md`
- Implementation Guide: `docs/SECURITY_IMPLEMENTATION_GUIDE.md`

---

## 📚 References

- [Laravel Security Best Practices](https://laravel.com/docs/10.x/security)
- [OWASP Top 10](https://owasp.org/www-project-top-ten/)
- [Spatie Activity Log](https://spatie.be/docs/laravel-activitylog)
- ISO 27001 Guidelines
- GDPR Principles

---

**Last Updated:** Oktober 2025  
**Version:** 1.0  
**Status:** Production Ready

