/**
 * Enhanced Dashboard JavaScript
 * Provides interactive features and data visualization for the dashboard
 */

document.addEventListener('DOMContentLoaded', function() {
    // Display current date in header
    displayCurrentDate();
    
    // Initialize dashboard animations
    initializeAnimations();
    
    // Initialize statistics with real data from backend
    initializeStats();
    
    // Initialize responsive behaviors
    initializeResponsiveBehaviors();
    
    // Add accessibility features
    enhanceAccessibility();
});

/**
 * Display formatted current date in the header
 */
function displayCurrentDate() {
    const dateElement = document.getElementById('current-date');
    if (dateElement) {
        const options = { weekday: 'long', year: 'numeric', month: 'long', day: 'numeric' };
        const today = new Date();
        dateElement.textContent = today.toLocaleDateString('id-ID', options);
    }
}

/**
 * Initialize animations for dashboard elements
 */
function initializeAnimations() {
    // Stagger fade-in animation for cards
    const cards = document.querySelectorAll('.card');
    cards.forEach((card, index) => {
        setTimeout(() => {
            card.classList.add('fade-in');
        }, 100 * index);
    });
    
    // Add hover effects to dashboard cards
    const dashboardCards = document.querySelectorAll('.dashboard-card');
    dashboardCards.forEach(card => {
        card.addEventListener('mouseenter', function() {
            this.style.transform = 'translateY(-8px)';
        });
        
        card.addEventListener('mouseleave', function() {
            this.style.transform = 'translateY(0)';
        });
    });
}

/**
 * Initialize statistics from backend data
 * Uses real data passed from Laravel controller
 */
function initializeStats() {
    // Get real data from backend (passed via window.dashboardStats)
    const stats = window.dashboardStats || {
        petani: 0,
        desa: 0,
        poktan: 0,
        kios: 0
    };
    
    // Animate counting for statistics
    animateCounter('total-petani', stats.petani);
    animateCounter('stat-petani', stats.petani);
    animateCounter('stat-desa', stats.desa);
    animateCounter('stat-poktan', stats.poktan);
    animateCounter('stat-kios', stats.kios);
}

/**
 * Animate counting up to a target number
 */
function animateCounter(elementId, targetNumber) {
    const element = document.getElementById(elementId);
    if (!element) return;
    
    const duration = 1500; // ms
    const frameDuration = 1000 / 60; // 60fps
    const totalFrames = Math.round(duration / frameDuration);
    let frame = 0;
    
    const counter = setInterval(() => {
        frame++;
        const progress = frame / totalFrames;
        const currentCount = Math.round(targetNumber * progress);
        
        element.textContent = currentCount.toLocaleString('id-ID');
        
        if (frame === totalFrames) {
            clearInterval(counter);
        }
    }, frameDuration);
}

/**
 * Initialize responsive behaviors for mobile optimization
 */
function initializeResponsiveBehaviors() {
    // Check if we're on a mobile device
    const isMobile = window.innerWidth < 768;
    
    // Adjust UI based on device type
    if (isMobile) {
        // Simplify some UI elements on mobile
        document.querySelectorAll('.dashboard-card-content p').forEach(p => {
            p.style.fontSize = '0.875rem';
        });
    }
    
    // Handle window resize events
    window.addEventListener('resize', function() {
        const currentIsMobile = window.innerWidth < 768;
        if (currentIsMobile !== isMobile) {
            // Reload page or adjust UI when switching between mobile/desktop
            // For this demo we'll just log it
            console.log('Viewport size changed significantly');
        }
    });
}

/**
 * Enhance accessibility features
 */
function enhanceAccessibility() {
    // Add ARIA labels where needed
    document.querySelectorAll('.dashboard-card').forEach((card, index) => {
        card.setAttribute('aria-label', `Menu card ${index + 1}`);
    });
    
    // Make sure all interactive elements are keyboard accessible
    document.querySelectorAll('.btn, .dashboard-card').forEach(element => {
        if (!element.getAttribute('tabindex')) {
            element.setAttribute('tabindex', '0');
        }
        
        // Add keyboard event listeners for non-button/link elements
        if (element.classList.contains('dashboard-card')) {
            element.addEventListener('keydown', function(e) {
                if (e.key === 'Enter' || e.key === ' ') {
                    e.preventDefault();
                    this.click();
                }
            });
        }
    });
}